"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const express_1 = require("express");
const express_validator_1 = require("express-validator");
const controller_1 = require("./controller");
const auth_1 = require("@/middlewares/auth");
const validation_1 = require("@/middlewares/validation");
const router = (0, express_1.Router)();
// Validation rules
const onboardingValidation = [
    (0, express_validator_1.body)('business_name')
        .isLength({ min: 2, max: 100 })
        .withMessage('Business name must be between 2 and 100 characters'),
    (0, express_validator_1.body)('profile_picture_url')
        .optional()
        .isURL()
        .withMessage('Profile picture must be a valid URL')
];
const profileUpdateValidation = [
    (0, express_validator_1.body)('business_name')
        .optional()
        .isLength({ min: 2, max: 100 })
        .withMessage('Business name must be between 2 and 100 characters'),
    (0, express_validator_1.body)('profile_picture_url')
        .optional()
        .isURL()
        .withMessage('Profile picture must be a valid URL'),
    (0, express_validator_1.body)('subscription_plan')
        .optional()
        .isIn(['free', 'basic', 'premium', 'enterprise'])
        .withMessage('Invalid subscription plan')
];
/**
 * @route   GET /api/vendors/me
 * @desc    Get current vendor profile
 * @access  Private (Vendor)
 */
router.get('/me', auth_1.authenticateUser, auth_1.requireVendor, controller_1.vendorController.getMyProfile);
/**
 * @route   POST /api/vendors/onboarding
 * @desc    Complete vendor onboarding
 * @access  Private (Vendor)
 */
router.post('/onboarding', auth_1.authenticateUser, auth_1.requireVendor, onboardingValidation, validation_1.validateRequest, controller_1.vendorController.completeOnboarding);
/**
 * @route   PUT /api/vendors/profile
 * @desc    Update vendor profile
 * @access  Private (Vendor)
 */
router.put('/profile', auth_1.authenticateUser, auth_1.requireVendor, profileUpdateValidation, validation_1.validateRequest, controller_1.vendorController.updateProfile);
/**
 * @route   GET /api/vendors/:id
 * @desc    Get vendor by ID
 * @access  Private (Admin)
 */
router.get('/:id', auth_1.authenticateUser, auth_1.requireAdmin, controller_1.vendorController.getVendorById);
/**
 * @route   GET /api/vendors
 * @desc    Get all vendors
 * @access  Private (Admin)
 */
router.get('/', auth_1.authenticateUser, auth_1.requireAdmin, controller_1.vendorController.getAllVendors);
exports.default = router;
//# sourceMappingURL=routes.js.map